from __future__ import annotations
import json
import base64
import sys
from urllib.parse import urlparse
from io import BytesIO
from PIL import Image

import requests

from typing import Any, Callable, Dict, Union

FileValue = Union[str, Callable[[], str]]

def resolve_entry(entry: Dict[str, Any]) -> Dict[str, str]:
    """
    Returns a normalized resource dict with:
      - path: str
      - file: str (base64)
    Accepts:
      - entry["file"] as base64 str OR callable returning base64 str
      - entry["url"] (will be fetched+encoded lazily)
    """
    path = entry["path"]

    if "file" in entry:
        f: FileValue = entry["file"]
        b64 = f() if callable(f) else f
        return {"path": path, "file": b64}

    if "url" in entry:
        return entry

    raise KeyError("Entry must contain either 'file' or 'url'.")


def url_to_base64(url: str) -> str:
	user_agent = "ScorecardGenerator/2.0 (mailto:ariell.zimran@gmail.com)"
	parsed = urlparse(url)
	if parsed.scheme not in ("http", "https"):
		raise ValueError(f"URL must be http/https: {url}")

	resp = requests.get(
		url,
		headers={"User-Agent": user_agent},
		timeout=30,
	)
	resp.raise_for_status()

	ctype = (resp.headers.get("Content-Type") or "").lower()
	if "image/" not in ctype and "application/octet-stream" not in ctype:
		raise ValueError(f"Response does not look like an image. Content-Type={ctype!r}")

	return base64.b64encode(resp.content).decode("ascii")
	
def jpg_url_to_png_base64(url: str) -> str:
    user_agent = "ScorecardGenerator/2.0 (mailto:ariell.zimran@gmail.com)"

    parsed = urlparse(url)
    if parsed.scheme not in ("http", "https"):
        raise ValueError(f"URL must be http/https: {url}")

    resp = requests.get(
        url,
        headers={"User-Agent": user_agent},
        timeout=30,
    )
    resp.raise_for_status()

    ctype = (resp.headers.get("Content-Type") or "").lower()
    if "image/jpeg" not in ctype and "image/jpg" not in ctype:
        raise ValueError(f"Response does not look like a JPEG. Content-Type={ctype!r}")

    # Load JPEG from memory
    with Image.open(BytesIO(resp.content)) as img:
        # JPEG has no alpha; convert to RGB
        img = img.convert("RGB")

        png_buffer = BytesIO()
        img.save(png_buffer, format="PNG")
        png_bytes = png_buffer.getvalue()

    return base64.b64encode(png_bytes).decode("ascii")
	
def gif_url_to_png_base64(url: str) -> str:
    user_agent = "ScorecardGenerator/2.0 (mailto:ariell.zimran@gmail.com)"

    parsed = urlparse(url)
    if parsed.scheme not in ("http", "https"):
        raise ValueError(f"URL must be http/https: {url}")

    resp = requests.get(
        url,
        headers={"User-Agent": user_agent},
        timeout=30,
    )
    resp.raise_for_status()

    ctype = (resp.headers.get("Content-Type") or "").lower()
    if "image/gif" not in ctype:
        raise ValueError(f"Response does not look like a GIF. Content-Type={ctype!r}")

    # Load GIF from memory
    with Image.open(BytesIO(resp.content)) as img:
        # Convert to a static RGB image (first frame if animated)
        img = img.convert("RGBA")

        png_buffer = BytesIO()
        img.save(png_buffer, format="PNG")
        png_bytes = png_buffer.getvalue()

    return base64.b64encode(png_bytes).decode("ascii")

def mlb_package(game):
	base_graphics=[
			{
				"path": "American League.png",
				"url": "https://content.sportslogos.net/logos/53/488/full/_american_league_logo_primary_20193833.png"
			},
			{
				"path": "National League.png",
				"url": "https://content.sportslogos.net/logos/54/489/full/9184__national_league-primary-2019.png"
			},
			{
				"path": "MLB.png",
				"url": "https://content.sportslogos.net/logos/4/490/full/4227__major_league_baseball-primary-2019.png"
			}
			]
			
	team_graphics = {
		"Minnesota Twins": 
			{
				"path": "Minnesota Twins.png",
				"url": "https://content.sportslogos.net/logos/53/65/full/minnesota_twins_logo_primary_2023_sportslogosnet-3953.png"
			},
		"Athletics":
			{
				"path": "Athletics.png",
				"url": "https://content.sportslogos.net/logos/53/6921/full/athletics__logo_primary_2025_sportslogosnet-5001.png"
			},
		"Oakland Athletics":
			{
				"path": "Oakland Athletics.png",
				"url": "https://content.sportslogos.net/logos/53/6921/full/athletics__logo_primary_2025_sportslogosnet-5001.png"
			},
		"Baltimore Orioles":
			{
				"path": "Baltimore Orioles.png",
				"url": "https://content.sportslogos.net/logos/53/52/full/baltimore_orioles_logo_primary_20195398.png"
			},
		"Boston Red Sox":
			{
				"path": "Boston Red Sox.png",
				"url": "https://content.sportslogos.net/logos/53/53/full/dxd6fu06fzn9mok79qsjnewjv.png"
			},
		"Chicago White Sox":	
			{
				"path": "Chicago White Sox.png",
				"url": "https://content.sportslogos.net/logos/53/55/full/chicago_white_sox_logo_primary_19911413.png"
			},
		"Cleveland Guardians":			
			{
				"path": "Cleveland Guardians.png",
				"url": "https://content.sportslogos.net/logos/53/6804/full/cleveland_guardians_logo_primary_20227577.png"
			},
		"Detroit Tigers":	
			{
				"path": "Detroit Tigers.png",
				"url": "https://content.sportslogos.net/logos/53/59/full/detroit_tigers_logo_primary_20162109.png"
			},
		"Houston Astros":				
			{
				"path": "Houston Astros.png",
				"url": "https://content.sportslogos.net/logos/53/4929/full/6753_houston_astros-alternate-2013.png"
			},
		"Kansas City Royals":		
			{
				"path": "Kansas City Royals.png",
				"url": "https://content.sportslogos.net/logos/53/62/full/kansas_city_royals_logo_primary_20198736.png"
			},
		"Los Angeles Angels":
			{
				"path": "Los Angeles Angels.png",
				"url": "https://content.sportslogos.net/logos/53/6521/full/4389_los_angeles_angels-primary-2016.png"
			},
		"New York Yankees":		
			{
				"path": "New York Yankees.png",
				"url": "https://content.sportslogos.net/logos/53/68/full/new_york_yankees_logo_primary_19685115.png"
			},
		"Seattle Mariners":		
			{
				"path": "Seattle Mariners.png",
				"url": "https://content.sportslogos.net/logos/53/75/full/seattle_mariners_logo_primary_19933809.png"
			},
		"Tampa Bay Rays":	
			{
				"path": "Tampa Bay Rays.png",
				"url": "https://content.sportslogos.net/logos/53/2535/full/tampa_bay_rays_logo_primary_20196768.png"
			},
		"Texas Rangers":	
			{
				"path": "Texas Rangers.png",
				"url": "https://content.sportslogos.net/logos/53/77/full/texas_rangers_logo_primary_2024_sportslogosnet-5505.png"
			},
		"Toronto Blue Jays":				
			{
				"path": "Toronto Blue Jays.png",
				"url": "https://content.sportslogos.net/logos/53/78/full/toronto_blue_jays_logo_primary_20208446.png"
			},
		"Cleveland Indians":		
			{
				"path": "Cleveland Indians.png",
				"url": "https://content.sportslogos.net/logos/53/57/full/5347_cleveland_indians-primary-2014.png"
			},
		"Arizona Diamondbacks":		
			{
				"path": "Arizona Diamondbacks.png",
				"url": "https://content.sportslogos.net/logos/54/50/full/arizona_diamondbacks_logo_primary_2024_sportslogosnet-3125.png"
			},
		"Atlanta Braves":		
			{
				"path": "Atlanta Braves.png",
				"url": "https://content.sportslogos.net/logos/54/51/full/atlanta_braves_logo_primary_20221869.png"
			},
		"Chicago Cubs":		
			{
				"path": "Chicago Cubs.png",
				"url": "https://content.sportslogos.net/logos/54/54/full/chicago_cubs_logo_primary_19792956.png"
			},
		"Cincinnati Reds":		
			{
				"path": "Cincinnati Reds.png",
				"url": "https://content.sportslogos.net/logos/54/56/full/cincinnati_reds_logo_primary_20133208.png"
			},
		"Colorado Rockies":
			{
				"path": "Colorado Rockies.png",
				"url": "https://content.sportslogos.net/logos/54/58/full/colorado_rockies_logo_primary_20171892.png"
			},
		"Los Angeles Dodgers":		
			{
				"path": "Los Angeles Dodgers.png",
				"url": "https://content.sportslogos.net/logos/54/63/full/los_angeles_dodgers_logo_primary_19792530.png"
			},
		"Miami Marlins":		
			{
				"path": "Miami Marlins.png",
				"url": "https://content.sportslogos.net/logos/54/3637/full/miami_marlins_logo_primary_20194007.png"
			},
		"Milwaukee Brewers":		
			{
				"path": "Milwaukee Brewers.png",
				"url": "https://content.sportslogos.net/logos/54/64/full/6474_milwaukee_brewers-primary-2020.png"
			},
		"New York Mets":		
			{
				"path": "New York Mets.png",
				"url": "https://content.sportslogos.net/logos/54/67/full/new_york_mets_logo_primary_1999sportslogosnet8711.png"
			},
		"Philadelphia Phillies":	
			{
				"path": "Philadelphia Phillies.png",
				"url": "https://content.sportslogos.net/logos/54/70/full/philadelphia_phillies_logo_primary_20193931.png"
			},
		"Pittsburgh Pirates":	
			{
				"path": "Pittsburgh Pirates.png",
				"url": "https://content.sportslogos.net/logos/54/71/full/1250_pittsburgh_pirates-primary-2014.png"
			},
		"San Diego Padres":		
			{
				"path": "San Diego Padres.png",
				"url": "https://content.sportslogos.net/logos/54/73/full/7517_san_diego_padres-primary-2020.png"
			},
		"San Francisco Giants":		
			{
				"path": "San Francisco Giants.png",
				"url": "https://content.sportslogos.net/logos/54/74/full/san_francisco_giants_logo_primary_20002208.png"
			},
		"St. Louis Cardinals":	
			{
				"path": "St. Louis Cardinals.png",
				"url": "https://content.sportslogos.net/logos/54/72/full/3zhma0aeq17tktge1huh7yok5.png"
			},
		"Washington Nationals":			
			{
				"path": "Washington Nationals.png",
				"url": "https://content.sportslogos.net/logos/54/578/full/washington_nationals_logo_primary_20117280.png"
			},		
		"Montreal Expos":
			{
				"path": "Montreal Expos.png",
				"url": "https://content.sportslogos.net/logos/54/66/full/w5aqbkrk24pv86dfgh5a1bh06.png"
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack
	
def higha_package(game):
	base_graphics=[
			{
				"path": "Northwest League.png",
				"url": "https://content.sportslogos.net/logos/47/3636/full/_northwest_league_logo_primary_20179542.png"
			},
			{
				"path": "MILB.png",
				"url": "https://content.sportslogos.net/logos/173/5231/full/minor-league-baseball-logo-primary-2024-523111012024.png"	
			},
			{
				"path": "Midwest League.png",
				"url": "https://content.sportslogos.net/logos/44/2603/full/_midwest_league_logo_primary_20175846.png"
			},			
			{
				"path": "South Atlantic League.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/45/2759/full/dsp4axugdxs3v8fntyagc4ukt.gif")
			}			
			]
			
	team_graphics = {
		"Eugene Emeralds": 
			{
				"path": "Eugene Emeralds.png",
				"url": "https://content.sportslogos.net/logos/47/2734/full/2272_eugene_emeralds-primary-2013.png"
			},
		"Everett AquaSox":
			{
				"path": "Everett AquaSox.png",
				"url": "https://content.sportslogos.net/logos/47/2735/full/7707_everett_aquasox-primary-2013.png"
			},
		"Hillsboro Hops":
			{
				"path": "Hillsboro Hops.png",
				"url": "https://content.sportslogos.net/logos/47/4893/full/586_hillsboro-hops-primary-2013.png"
			},
		"Vancouver Canadians":
			{
				"path": "Vancouver Canadians.png",
				"url": "https://content.sportslogos.net/logos/47/2739/full/1783_vancouver_canadians-primary-2014.png"
			},
		"Spokane Indians":
			{
				"path": "Spokane Indians.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/47/2737/full/gq2csfe24vrdfg4hmljf2domn.gif")
			},
		"Tri-City Dust Devils":
			{
				"path": "Tri-City Dust Devils.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/47/2738/full/umelyjnf7earb5e4hewjpdcgp.gif")
			},
		"Beloit Sky Carp":
			{
				"path": "Beloit Sky Carp.png",
				"url": "https://content.sportslogos.net/logos/44/6953/full/beloit-sky-carp-logo-primary-2022-695332222022.png"
			},
		"Great Lakes Loons":
			{
				"path": "Great Lakes Loons.png",
				"url": "https://content.sportslogos.net/logos/44/2595/full/4703_great_lakes_loons-alternate-2016.png"
			},
		"Lake County Captains":
			{
				"path": "Lake County Captains.png",
				"url": "https://content.sportslogos.net/logos/44/3023/full/lake_county_captains_logo_primary_2025_sportslogosnet-5006.png"
			},
		"Quad Cities River Bandits":
			{
				"path": "Quad Cities River Bandits.png",
				"url": "https://content.sportslogos.net/logos/44/2600/full/5282_quad_cities_river_bandits-alternate-2014.png"
			},
		"South Bend Cubs":
			{
				"path": "South Bend Cubs.png",
				"url": "https://content.sportslogos.net/logos/44/5547/full/3364_south_bend_cubs-primary-2015.png"
			},
		"Cedar Rapids Kernels":
			{
				"path": "Cedar Rapids Kernels.png",
				"file": lambda: gif_url_to_png_base64("https%3A%2F%2Fcontent.sportslogos.net%2Flogos%2F44%2F2591%2Ffull%2Fm4ca3x1qzg0feghpfr3eknaap.gif")
			},
		"Dayton Dragons":
			{
				"path": "Dayton Dragons.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/44/2593/full/sho1ejdiif1ye0s80hrvfa1nl.gif")
			},
		"Fort Wayne TinCaps":
			{
				"path": "Fort Wayne TinCaps.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/44/2727/full/elusensaayeeediiescrbhfx2.gif")
			},
		"Lansing Lugnuts":
			{
				"path": "Lansing Lugnuts.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/44/2597/full/i1pfr8gslm1863vf30cgd7lll.gif")
			},
		"Peoria Chiefs":
			{
				"path": "Peoria Chiefs.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/44/2598/full/19chp7iphkg6fhehxo0ssx6dn.gif")
			},
		"West Michigan Whitecaps":
			{
				"path": "West Michigan Whitecaps.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/44/2601/full/1i7hvflgpglkhi3yk999keq8f.gif")
			},
		"Wisconsin Timber Rattlers":
			{
				"path": "Wisconsin Timber Rattlers.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/44/2602/full/xzjaqyg3masg5x4d9ng7sooa9.gif")
			},
		"Asheville Tourists":
			{
				"path": "Asheville Tourists.png",
				"url": "https://content.sportslogos.net/logos/45/2742/full/asheville_tourists_logo_primary_20115856.png"
			},
		"Bowling Green Hot Rods":
			{
				"path": "Bowling Green Hot Rods.png",
				"url": "https://content.sportslogos.net/logos/45/6840/full/bowling_green_hot_rods_logo_primary_2021_sportslogosnet-1926.png"
			},
		"Brooklyn Cyclones":
			{
				"path": "Brooklyn Cyclones.png",
				"url": "https://content.sportslogos.net/logos/45/6835/full/brooklyn_cyclones_logo_primary_2021_sportslogosnet-6054.png"
			},
		"Frederick Keys":
			{
				"path": "Frederick Keys.png",
				"url": "https://content.sportslogos.net/logos/45/6954/full/frederick-keys-logo-primary-2026-695497472026.png"
			},
		"Greensboro Grashoppers":
			{
				"path": "Greensboro Grashoppers.png",
				"url": "https://content.sportslogos.net/logos/45/2751/full/greensboro_grasshoppers_logo_primary_20055893.png"
			},
		"Greenville Drive":
			{
				"path": "Greenville Drive.png",
				"url": "https://content.sportslogos.net/logos/45/2746/full/greenville_drive_logo_primary_20067714.png"
			},
		"Hub City Spartanburgers":
			{
				"path": "Hub City Spartanburgers.png",
				"url": "https://content.sportslogos.net/logos/45/6955/full/hub-city-spartanburgers-logo-primary-2025-695570532025.png"
			},
		"Hudson Valley Renegades":
			{
				"path": "Hudson Valley Renegades.png",
				"url": "https://content.sportslogos.net/logos/45/6836/full/hudson_valley_renegades_logo_alternate_2021_sportslogosnet-7935.png"
			},
		"Jersey Shore Blueclaws":
			{
				"path": "Jersey Shore Blueclaws.png",
				"url": "https://content.sportslogos.net/logos/45/6837/full/jersey_shore_blueclaws_logo_primary_2021_sportslogosnet-9772.png"
			},
		"Rome Emperors":
			{
				"path": "Rome Emperors.png",
				"url": "https://content.sportslogos.net/logos/45/6956/full/rome-emperors-alternate-logo-2024-695652712024.png"
			},
		"Wilmington Blue Rocks":
			{
				"path": "Wilmington Blue Rocks.png",
				"url": "https://content.sportslogos.net/logos/45/6838/full/wilmington_blue_rocks_logo_primary_2021_sportslogosnet-8264.png"
			},
		"Winston-Salem Dash":
			{
				"path": "Winston-Salem Dash.png",
				"url": "https://content.sportslogos.net/logos/45/6839/full/winston-salem_dash_logo_primary_2021_sportslogosnet-5371.png"
			},
		"Aberdeen Iron Birds":
			{
				"path": "Aberdeen Iron Birds.png",
				"url": "https://content.sportslogos.net/logos/45/6834/full/aberdeen_ironbirds_logo_primary_20226922.png"
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack

def aaa_package(game):
	base_graphics=[
			{
				"path": "International League.png",
				"url": "https://content.sportslogos.net/logos/36/938/full/_international_league_logo_primary_2008_sportslogosnet-5036.png"
			},
			{
				"path": "MILB.png",
				"url": "https://content.sportslogos.net/logos/173/5231/full/minor-league-baseball-logo-primary-2024-523111012024.png"	
			},
			{
				"path": "Pacific Coast League.png",
				"url": "https://content.sportslogos.net/logos/37/1028/full/_pacific_coast_league_logo_primary_2022_sportslogosnet-5642.png"
			}			
			]
			
	team_graphics = {
		"Tacoma Rainiers": 
			{
				"path": "Tacoma Rainiers.png",
				"url": "https://content.sportslogos.net/logos/37/1027/full/5248_tacoma_rainiers-secondary-2015.png"
			},
		"Sugar Land Space Cowboys":
			{
				"path": "Sugar Land Space Cowboys.png",
				"url": "https://content.sportslogos.net/logos/37/6901/full/sugar_land_space_cowboys_logo_primary_2021_sportslogosnet-2655.png"
			},
		"Salt Lake Bees":
			{
				"path": "Salt Lake Bees.png",
				"url": "https://content.sportslogos.net/logos/37/1131/full/7649_salt_lake_bees-alternate-2015.png"
			},
		"Sacramento River Cats":
			{
				"path": "Sacramento River Cats.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/37/1025/full/e6psr5i6hhj5ajvqyide.gif")
			},
		"Round Rock Express":
			{
				"path": "Round Rock Express.png",
				"url": "https://content.sportslogos.net/logos/37/1024/full/round_rock_express_logo_primary_20193334.png"
			},
		"Reno Aces":
			{
				"path": "Reno Aces.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/47/2738/full/umelyjnf7earb5e4hewjpdcgp.gif")
			},
		"Oklahoma City Comets":
			{
				"path": "Oklahoma City Comets.png",
				"url": "https://content.sportslogos.net/logos/37/6923/full/oklahoma_city_comets_logo_primary_2025_sportslogosnet-2539.png"
			},
		"Las Vegas Aviators":
			{
				"path": "Las Vegas Aviators.png",
				"url": "https://content.sportslogos.net/logos/37/6667/full/las_vegas_aviators_logo_primary_2023_sportslogosnet-5770.png"
			},
		"El Paso Chihuahuas":
			{
				"path": "El Paso Chihuahuas.png",
				"url": "https://content.sportslogos.net/logos/37/5076/full/3110_el_paso_chihuahuas-primary-2014.png"
			},
		"Albuquerque Isotopes":
			{
				"path": "Albuquerque Isotopes.png",
				"url": "https://content.sportslogos.net/logos/37/1013/full/albuquerque_isotopes_logo_primary_2023_sportslogosnet-6442.png"
			},
		"Oklahoma City Baseball Club":
			{
				"path": "Oklahoma City Baseball Club.png",
				"url": "https://content.sportslogos.net/logos/37/6900/full/oklahoma_city_baseball_club_logo_primary_2024_sportslogosnet-5944.png"
			},
		"Worcester Red Sox":
			{
				"path": "Worcester Red Sox.png",
				"url": "https://content.sportslogos.net/logos/36/6851/full/worcester_red_sox_logo_primary_2021_sportslogosnet-2601.png"
			},
		"Toledo Mud Hens":
			{
				"path": "Toledo Mud Hens.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/36/937/full/794wsl5qbo9qicg4hvj19nbt3.gif")
			},
		"Syracuse Mets":
			{
				"path": "Syracuse Mets.png",
				"url": "https://content.sportslogos.net/logos/36/6627/full/syracuse_mets_logo_primary_20198089.png"
			},
		"St. Paul Saints":
			{
				"path": "St. Paul Saints.png",
				"url": "https://content.sportslogos.net/logos/166/5004/full/3058_st_paul__saints-secondary-2006.png"
			},
		"Scranton/Wilkes-Barre RailRiders":
			{
				"path": "Scranton/Wilkes-Barre RailRiders.png",
				"url": "https://content.sportslogos.net/logos/36/4937/full/1737_scranton_railriders-primary-2013.png"
			},
		"Rochester Red Wings":
			{
				"path": "Rochester Red Wings.png",
				"url": "https://content.sportslogos.net/logos/36/934/full/6254_rochester_red_wings-primary-2014.png"
			},
		"Omaha Storm Chasers":
			{
				"path": "Omaha Storm Chasers.png",
				"url": "https://content.sportslogos.net/logos/36/6874/full/omaha_storm_chasers_logo_primary_2021_sportslogosnet-9708.png"
			},
		"Norfolk Tides":
			{
				"path": "Norfolk Tides.png",
				"url": "https://content.sportslogos.net/logos/36/930/full/norfolk-tides-logo-primary-2016-2883.png"
			},
		"Nashville Sounds":
			{
				"path": "Nashville Sounds.png",
				"url": "https://content.sportslogos.net/logos/36/6873/full/nashville_sounds_logo_primary_2021_sportslogosnet-7206.png"
			},
		"Memphis Redbirds":
			{
				"path": "Memphis Redbirds.png",
				"url": "https://content.sportslogos.net/logos/36/6871/full/memphis_redbirds_logo_primary_2021_sportslogosnet-7129.png"
			},
		"Louisville Bats":
			{
				"path": "Louisville Bats.png",
				"url": "https://content.sportslogos.net/logos/36/929/full/1853_louisville_bats-alternate-2016.png"
			},
		"Lehigh Valley IronPigs":
			{
				"path": "Lehigh Valley IronPigs.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/36/2548/full/j0nleuh257g5ewoykfrb44oz6.gif")
			},
		"Jacksonville Jumbo Shrimp":
			{
				"path": "Jacksonville Jumbo Shrimp.png",
				"url": "https://content.sportslogos.net/logos/36/6850/full/jacksonville_jumbo_shrimp_logo_primary_2021_sportslogosnet-8427.png"
			},
		"Iowa Cubs":
			{
				"path": "Iowa Cubs.png",
				"url": "https://content.sportslogos.net/logos/36/6870/full/iowa_cubs_logo_primary_2021_sportslogosnet-7790.png"
			},
		"Indianapolis Indians":
			{
				"path": "Indianapolis Indians.png",
				"url": "https://content.sportslogos.net/logos/36/928/full/indianapolis-indians-logo-primary-2026-92893142026.png"
			},
		"Gwinnett Stripers":
			{
				"path": "Gwinnett Stripers.png",
				"url": "https://content.sportslogos.net/logos/36/6650/full/9276_gwinnett_stripers-primary-2018.png"
			},
		"Durham Bulls":
			{
				"path": "Durham Bulls.png",
				"url": "https://content.sportslogos.net/logos/36/927/full/4516.png"
			},
		"Columbus Clippers":
			{
				"path": "Columbus Clippers.png",
				"url": "https://content.sportslogos.net/logos/36/926/full/columbus_clippers_logo_primary_20097787.png"
			},
		"Charlotte Knights":
			{
				"path": "Charlotte Knights.png",
				"url": "https://content.sportslogos.net/logos/36/925/full/charlotte_knights_logo_primary_2023_sportslogosnet-2592.png"
			},
		"Buffalo Bisons":
			{
				"path": "Buffalo Bisons.png",
				"url": "https://content.sportslogos.net/logos/36/924/full/buffalo_bisons_logo_primary_20134771.png"
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack
	
def aa_package(game):
	base_graphics=[
			{
				"path": "Texas League.png",
				"url": "https://content.sportslogos.net/logos/40/1432/full/5948__texas_league-primary-2016.png"
			},
			{
				"path": "MILB.png",
				"url": "https://content.sportslogos.net/logos/173/5231/full/minor-league-baseball-logo-primary-2024-523111012024.png"	
			},
			{
				"path": "Southern League.png",
				"url": "https://content.sportslogos.net/logos/39/1317/full/4733__southern_league-primary-2016.png"
			},
			{
				"path": "Eastern League.png",
				"url": "https://content.sportslogos.net/logos/38/1049/full/_eastern_league_logo_primary_20195144.png"			
			}
			]
			
	team_graphics = {
		"Wichita Wind Surge": 
			{
				"path": "Wichita Wind Surge.png",
				"url": "https://content.sportslogos.net/logos/40/6897/full/wichita_wind_surge_logo_primary_2021_sportslogosnet-5108.png"
			},
		"Tulsa Drillers":
			{
				"path": "Tulsa Drillers.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/40/1446/full/6671.gif")
			},
		"Springfield Cardinals":
			{
				"path": "Springfield Cardinals.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/40/1452/full/6666.gif")
			},
		"San Antonio Missions":
			{
				"path": "San Antonio Missions.png",
				"url": "https://content.sportslogos.net/logos/40/6898/full/san_antonio_missions_logo_primary_2021sportslogosnet6847.png"
			},
		"Frisco RoughRiders":
			{
				"path": "Frisco RoughRiders.png",
				"url": "https://content.sportslogos.net/logos/40/1435/full/8114_frisco_roughriders-alternate-2015.png"
			},
		"Midland RockHounds":
			{
				"path": "Midland RockHounds.png",
				"url": "https://content.sportslogos.net/logos/40/1438/full/midland_rockhounds_logo_primary_2022sportslogosnet1570.png"
			},
		"Northwest Arkansas Naturals":
			{
				"path": "Northwest Arkansas Naturals.png",
				"url": "https://content.sportslogos.net/logos/40/2549/full/northwest_arkansas_naturals_logo_primary_2024_sportslogosnet-6409.png"
			},
		"Corpus Christi Hooks":
			{
				"path": "Corpus Christi Hooks.png",
				"url": "https://content.sportslogos.net/logos/40/1440/full/corpus_christi_hooks_logo_primary_2025sportslogosnet2113.png"
			},
		"Arkansas Travelers":
			{
				"path": "Arkansas Travelers.png",
				"url": "https://content.sportslogos.net/logos/40/1433/full/6379_arkansas_travelers-primary-2014.png"
			},
		"Amarillo Sod Poodles":
			{
				"path": "Amarillo Sod Poodles.png",
				"url": "https://content.sportslogos.net/logos/37/1013/full/albuquerque_isotopes_logo_primary_2023_sportslogosnet-6442.png"
			},
		"Rocket City Trash Pandas":
			{
				"path": "Rocket City Trash Pandas.png",
				"url": "https://content.sportslogos.net/logos/39/6934/full/rocket_city_trash_pandas_logo_primary_2021sportslogosnet5742.png"
			},
		"Pensacola Blue Wahoos":
			{
				"path": "Pensacola Blue Wahoos.png",
				"url": "https://content.sportslogos.net/logos/39/3874/full/8161_pensacola_blue_wahoos-secondary-2012.png"
			},
		"Montgomery Biscuits":
			{
				"path": "Montgomery Biscuits.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/39/1327/full/2xvgg9vsbldj705t8ffh3mf9h.gif")
			},
		"Knoxville Smokies":
			{
				"path": "Knoxville Smokies.png",
				"url": "https://content.sportslogos.net/logos/39/6936/full/knoxville_smokies_logo_primary_2025sportslogosnet2041.png"
			},
		"Tennessee Smokies":
			{
				"path": "Tennessee Smokies.png",
				"url": "https://content.sportslogos.net/logos/39/6936/full/knoxville_smokies_logo_primary_2025sportslogosnet2041.png"
			},
		"Columbus Clingstones":
			{
				"path": "Columbus Clingstones.png",
				"url": "https://content.sportslogos.net/logos/39/6935/full/columbus_clingstones_logo_primary_2025sportslogosnet2326.png"
			},
		"Chattanooga Lookouts":
			{
				"path": "Chattanooga Lookouts.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/39/1320/full/01tytzwosbsethvhfr9w1gjcl.gif")
			},
		"Birmingham Barons":
			{
				"path": "Birmingham Barons.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/39/1318/full/gt51reql52oh60w48b16omna0.gif")
			},
		"Biloxi Shuckers":
			{
				"path": "Biloxi Shuckers.png",
				"url": "https://content.sportslogos.net/logos/39/5497/full/9270_biloxi_shuckers-primary-2015.png"
			},
		"Somerset Patriots":
			{
				"path": "Somerset Patriots.png",
				"url": "https://content.sportslogos.net/logos/38/6909/full/somerset_patriots_logo_primary_2021_sportslogosnet-7493.png"
			},
		"Richmond Flying Squirrels":
			{
				"path": "Richmond Flying Squirrels.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/38/2771/full/v7q7guo3sg2ans17ignn3eyi2.gif")
			},
		"Reading Fightin Phils":
			{
				"path": "Reading Fightin Phils.png",
				"url": "https://content.sportslogos.net/logos/38/4938/full/reading_fightin_phils_logo_primary_2019sportslogosnet2070.png"
			},
		"Portland Sea Dogs":
			{
				"path": "Portland Sea Dogs.png",
				"url": "https://content.sportslogos.net/logos/38/1040/full/portland-sea-dogs-logo-primary-2003-3414.png"
			},
		"New Hampshire Fisher Cats":
			{
				"path": "New Hampshire Fisher Cats.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/38/1039/full/949jk6f10bnrypp21whmfexf2.gif")
			},
		"Hartford Yard Goats":
			{
				"path": "Hartford Yard Goats.png",
				"url": "https://content.sportslogos.net/logos/38/5798/full/hartford_yard_goats_logo_primary_2016sportslogosnet7049.png"
			},
		"Harrisburg Senators":
			{
				"path": "Harrisburg Senators.png",
				"url": "https://content.sportslogos.net/logos/38/1037/full/harrisburg_senators_logo_primary_20066852.png"
			},
		"Erie SeaWolves":
			{
				"path": "Erie SeaWolves.png",
				"url": "https://content.sportslogos.net/logos/38/1036/full/8192_erie_seawolves-primary-2013.png"
			},
		"Chesapeake Baysox":
			{
				"path": "Chesapeake Baysox.png",
				"url": "https://content.sportslogos.net/logos/38/6928/full/chesapeake_baysox_logo_alternate_2025sportslogosnet5668.png"
			},
		"Bowie Baysox":
			{
				"path": "Bowie Baysox.png",
				"url": "https://content.sportslogos.net/logos/38/6928/full/chesapeake_baysox_logo_alternate_2025sportslogosnet5668.png"
			},
		"Binghamton Rumble Ponies":
			{
				"path": "Binghamton Rumble Ponies.png",
				"url": "https://content.sportslogos.net/logos/38/6442/full/binghamton_rumble_ponies_logo_primary_20175250.png"
			},
		"Columbus Clippers":
			{
				"path": "Columbus Clippers.png",
				"url": "https://content.sportslogos.net/logos/36/926/full/columbus_clippers_logo_primary_20097787.png"
			},
		"Altoona Curve":
			{
				"path": "Altoona Curve.png",
				"url": "https://content.sportslogos.net/logos/38/1033/full/altoona_curve_logo_primary_20116526.png"
			},
		"Akron RubberDucks":
			{
				"path": "Akron RubberDucks.png",
				"url": "https://content.sportslogos.net/logos/38/5141/full/akron_rubberducks_logo_alternate_2014_sportslogosnet-5152.png"
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack			

def lowa_package(game):
	base_graphics=[
			{
				"path": "Florida State League.png",
				"url": "https://content.sportslogos.net/logos/43/2574/full/_florida_state_league_logo_primary_2020_sportslogosnet-7192.png"
			},
			{
				"path": "MILB.png",
				"url": "https://content.sportslogos.net/logos/173/5231/full/minor-league-baseball-logo-primary-2024-523111012024.png"	
			},
			{
				"path": "Carolina League.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/42/2211/full/tzbf5lgftz0sg6400mfu.gif")
			},			
			{
				"path": "California League.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/41/2210/full/zag1rh2hfy5vrihmrhf6.gif")
			}			
			]
			
	team_graphics = {
		"Tampa Tarpons": 
			{
				"path": "Tampa Tarpons.png",
				"url": "https://content.sportslogos.net/logos/43/6654/full/2899_tampa_tarpons-primary-2018.png"
			},
		"St. Lucie Mets":
			{
				"path": "St. Lucie Mets.png",
				"url": "https://content.sportslogos.net/logos/43/2579/full/7170_st_lucie_mets-secondary-2013.png"
			},
		"Palm Beach Cardinals":
			{
				"path": "Palm Beach Cardinals.png",
				"url": "https://content.sportslogos.net/logos/43/2578/full/palm-beach-cardinals-logo-primary-2015-257886202015.png"
			},
		"Lakeland Flying Tigers":
			{
				"path": "Lakeland Flying Tigers.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/43/2584/full/exiemq8arf2hivcn089ezi0eh.gif")
			},
		"Jupiter Hammerheads":
			{
				"path": "Jupiter Hammerheads.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/43/2577/full/1gpale9af5uir7025ahyngkh2.gif")
			},
		"Fort Meyers Mighty Mussels":
			{
				"path": "Fort Meyers Mighty Mussels.png",
				"url": "https://content.sportslogos.net/logos/43/6702/full/4049_fort_myers_mighty_mussels-alternate-2020.png"
			},
		"Dunedin Blue Jays":
			{
				"path": "Dunedin Blue Jays.png",
				"url": "https://content.sportslogos.net/logos/43/2582/full/dunedin-blue-jays-logo-primary-2012-3071.png"
			},
		"Daytona Tortugas":
			{
				"path": "Daytona Tortugas.png",
				"url": "https://content.sportslogos.net/logos/43/5546/full/6859_daytona_tortugas-primary-2015.png"
			},
		"Clearwater Threshers":
			{
				"path": "Clearwater Thresers.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/43/2581/full/g4suxghs9mqgyinfsh1qhidlj.gif")
			},
		"Bradenton Marauders":
			{
				"path": "Bradenton Marauders.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/43/3035/full/rukj5af6v3ybmqycadxffg9o1.gif")
			},
		"Salem Red Sox":
			{
				"path": "Salem Red Sox.png",
				"url": "https://content.sportslogos.net/logos/42/2726/full/salem-red-sox-logo-primary-2009-1969.png"
			},
		"Down East Wood Ducks":
			{
				"path": "Down East Wood Ducks.png",
				"url": "https://content.sportslogos.net/logos/42/6669/full/4708_down_east_wood_ducks-primary-2017.png"
			},
		"Carolina Mudcats":
			{
				"path": "Carolina Mudcats.png",
				"url": "https://content.sportslogos.net/logos/42/5142/full/carolina-mudcats-logo-primary-2012-5024.png"
			},
		"Wilson Warbirds":
			{
				"path": "Wilson Warbirds.png",
				"url": "https://content.sportslogos.net/logos/42/6952/full/wilson-warbirds-logo-primary-2026-695293442026.png"
			},
		"Salem RidgeYaks":
			{
				"path": "Salem RidgeYaks.png",
				"url": "https://content.sportslogos.net/logos/42/6951/full/salem-ridgeyaks-logo-primary-2026-695173042026.png"
			},
		"Myrtle Beach Pelicans":
			{
				"path": "Myrtle Beach Pelicans.png",
				"url": "https://content.sportslogos.net/logos/42/2225/full/n0x6l9egbdrmgxye7kk7u347j.png"
			},
		"Lynchburg Hillcats":
			{
				"path": "Lynchburg Hillcats.png",
				"url": "https://content.sportslogos.net/logos/42/2219/full/7326_lynchburg_hillcats-alternate-2017.png"
			},
		"Kannapolis Cannon Ballers":
			{
				"path": "Kannapolis Cannon Ballers.png",
				"url": "https://content.sportslogos.net/logos/42/6943/full/kannapolis-cannon-ballers-logo-primary-2021-694337582021.png"
			},
		"Hickory Crawdads":
			{
				"path": "Hickory Crawdads.png",
				"url": "https://content.sportslogos.net/logos/42/6942/full/hickory-crawdads-logo-primary-2025-694231372025.png"
			},
		"Fredericksburg Nationals":
			{
				"path": "Fredericksburg Nationals.png",
				"url": "https://content.sportslogos.net/logos/42/6745/full/1446_fredericksburg_nationals-primary-2020.png"
			},
		"Fayetteville Woodpeckers":
			{
				"path": "Fayetteville Woodpeckers.png",
				"url": "https://content.sportslogos.net/logos/42/6668/full/6197_fayetteville_woodpeckers-primary-2019.png"
			},
		"Delmarva Shorebirds":
			{
				"path": "Delmarva Shorebirds.png",
				"url": "https://content.sportslogos.net/logos/42/6941/full/delmarva-shorebirds-logo-primary-2021-694172332021.png"
			},
		"Columbia Fireflies":
			{
				"path": "Columbia Fireflies.png",
				"url": "https://content.sportslogos.net/logos/42/6939/full/columbia-fireflies-logo-primary-2021-693926492021.png"
			},
		"Charleston RiverDogs":
			{
				"path": "Charleston RiverDogs.png",
				"url": "https://content.sportslogos.net/logos/42/6940/full/charleston-riverdogs-logo-primary-2021-694045742021.png"
			},
		"Augusta GreenJackets":
			{
				"path": "Augusta GreenJackets.png",
				"url": "https://content.sportslogos.net/logos/42/6938/full/augusta-greenjackets-logo-primary-2021-7260.png"
			},
		"Modesto Nuts":
			{
				"path": "Modesto Nuts.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/41/2147/full/h8cphkjujh0c9jjfgepw.gif")
			},
		"Visalia Rawhide":
			{
				"path": "Visalia Rawhide.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/41/2182/full/bgp1hh03ryinsbg73pfwgfgz6.gif")
			},
		"Stockton Ports":
			{
				"path": "Stockton Ports.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/41/2178/full/hd4fgg8eulfte3bh2gst.gif")
			},
		"San Jose Giants":
			{
				"path": "San Jose Giants.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/41/2174/full/4eegf706do5w6dethhudw3y8z.gif")
			},
		"Rancho Cucamonga Quakes":
			{
				"path": "Rancho Cucamonga Quakes.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/41/2150/full/klnemgs6tloo3a2ownsz.gif")
			},
		"Ontario Tower Buzzers":
			{
				"path": "Ontario Tower Buzzers.png",
				"url": "https://content.sportslogos.net/logos/41/6949/full/ontario-tower-buzzers-logo-primary-2026-694972432026.png"
			},
		"Lake Elsinore Storm":
			{
				"path": "Lake Elsinore Storm.png",
				"url": "https://content.sportslogos.net/logos/41/2138/full/lake_elsinore_storm_logo_primary_2024sportslogosnet8060.png"
			},
		"Inland Empire 66ers":
			{
				"path": "Inland Empire 66ers.png",
				"url": "https://content.sportslogos.net/logos/41/2166/full/inland-empire-66ers-logo-primary-2022-216617122022.png"
			},
		"Fresno Grizzlies":
			{
				"path": "Fresno Grizzlies.png",
				"url": "https://content.sportslogos.net/logos/41/6899/full/fresno_grizzlies_logo_primary_2021sportslogosnet7271.png"
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack
	
def lmb_package(game):
	base_graphics=[
			{
				"path": "Mexican League.png",
				"url": "https://content.sportslogos.net/logos/165/4976/full/6336_mexican_league_(liga_mexicana_de_beisbol)-secondary-2009.png"	
			},
			{
				"path": "LMB.png",
				"url": "https://content.sportslogos.net/logos/67/1820/full/mexico__logo_primary_20064949.png"
			}			
			]
			
	team_graphics = {
		"Leones de Yucatan": 
			{
				"path": "Leones de Yucatan.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/1/19/Leones_de_Yucatán_logo.svg/1280px-Leones_de_Yucatán_logo.svg.png")
			},
		"El Aguila de Veracruz":
			{
				"path": "El Aguila de Veracruz.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/9/9e/El_Águila_de_Veracruz.png")
			},
		"Olmecas de Tabasco":
			{
				"path": "Olmecas de Tabasco.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/6/69/Icono_olmecas_tabasco_2024.png")
			},
		"Saraperos de Saltillo":
			{
				"path": "Saraperos de Saltillo.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/c/c0/Saraperos_de_Saltillo_logo.png")
			},
		"Pericos de Puebla":
			{
				"path": "Pericos de Puebla.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/6/62/Pericos_de_Puebla_logo.svg/1280px-Pericos_de_Puebla_logo.svg.png")
			},
		"Tigres de Quintana Roo":
			{
				"path": "Tigres de Quintana Roo.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/9/9d/Tigres_Quintana_Roo_logo_%2810_stars%29.svg")
			},
		"Piratas de Campeche":
			{
				"path": "Piratas de Campeche.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/4/4e/PiratasdeCampeche.png")
			},
		"Guerreros de Oaxaca":
			{
				"path": "Guerreros de Oaxaca.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/1/1f/Guerreros-Oaxaca.png")
			},
		"Diablos Rojos del Mexico":
			{
				"path": "Diablos Rojos del Mexico.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/b/b5/Diablos-Mexico.png")
			},
		"Conspiradores de Queretaro":
			{
				"path": "Conspiradores de Queretaro.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/5/5a/Conspiradores_de_Querétaro_logo.svg/960px-Conspiradores_de_Querétaro_logo.svg.png")
			},
		"Bravos de Leon":
			{
				"path": "Bravos de Leon.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/2/28/Bravos_de_Leon_cap_logo.png")
			},
		"Toros de Tijuana":
			{
				"path": "Toros de Tijuana.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/8/8f/Tijuana_Toros_logo.png")
			},
		"Tecolotes de los dos Laredos":
			{
				"path": "Tecolotes de los dos Laredos.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/2/2d/Tecolotes_Dos_Laredos_logo.png")
			},
		"Sultanes de Monterrey":
			{
				"path": "Sultanes de Monterrey.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/3/30/Sultanes_de_Monterrey_logo.svg/1280px-Sultanes_de_Monterrey_logo.svg.png")
			},
		"Rieleros de Aguascalientes":
			{
				"path": "Rieleros de Aguascalientes.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/b/b6/Rieleros_de_Aguascalientes_logo.svg/1280px-Rieleros_de_Aguascalientes_logo.svg.png")
			},
		"Dorados de Chihuahua":
			{
				"path": "Dorados de Chihuahua.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/b/b2/Dorados_de_Chihuahua_baseball_logo.svg/1280px-Dorados_de_Chihuahua_baseball_logo.svg.png")
			},
		"Charros de Jalisco":
			{
				"path": "Charros de Jalisco.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/3/37/Charros_de_Jalisco_logo.svg/1280px-Charros_de_Jalisco_logo.svg.png")
			},
		"Caliente de Durango":
			{
				"path": "Caliente de Durango.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/3/32/Caliente_de_Durango_logo.svg/1280px-Caliente_de_Durango_logo.svg.png")
			},
		"Algodoneros Union Laguna":
			{
				"path": "Algodoneros Union Laguna.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/0/04/Algodoneros_de_Unión_Laguna_logo.png")
			},
		"Acereros del Norte":
			{
				"path": "Acereros del Norte.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/f/f4/Acereros-Monclova.png")
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack
	
def wbc_package(game):
	base_graphics=[
			{
				"path": "World Baseball Classic.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/2/2e/WBC_logo.svg/1920px-WBC_logo.svg.png")
			},
			{
				"path": "MLB.png",
				"url": "https://content.sportslogos.net/logos/4/490/full/4227__major_league_baseball-primary-2019.png"
			}			
			]
			
	team_graphics = {
		"Venezuela": 
			{
				"path": "Venezuela.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1830/full/vsuyxmzksfepgxmeepx6.gif")
			},
		"United States":
			{
				"path": "United States.png",
				"url": "https://content.sportslogos.net/logos/67/1822/full/united_states__logo_primary_2013_sportslogosnet-1403.png"
			},
		"South Africa":
			{
				"path": "South Africa.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1821/full/7s6qfu7vkgzxcm3b0ahv.gif")
			},
		"Puerto Rico":
			{
				"path": "Puerto Rico.png",
				"url": "https://content.sportslogos.net/logos/67/1826/full/puerto_rico__logo_primary_2013_sportslogosnet-7683.png"
			},
		"Panama":
			{
				"path": "Panama.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1825/full/ywvzkpxyzih3se6vq1ie.gif")
			},
		"Kingdom of the Netherlands":
			{
				"path": "Kingdom of the Netherlands.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1824/full/76h9gnhiu9hrnebulk5n.gif")
			},
		"Mexico":
			{
				"path": "Mexico.png",
				"url": "https://content.sportslogos.net/logos/67/1820/full/mexico__logo_primary_20064949.png"
			},
		"Korea":
			{
				"path": "Korea.png",
				"url": "https://content.sportslogos.net/logos/67/1818/full/korea__logo_primary_2023_sportslogosnet-6794.png"
			},
		"Japan":
			{
				"path": "Japan.png",
				"url": "https://content.sportslogos.net/logos/67/1817/full/japan__logo_primary_2017_sportslogosnet-9623.png"
			},
		"Italy":
			{
				"path": "Italy.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1829/full/7505.gif")
			},
		"Dominican Republic":
			{
				"path": "Dominican Republic.png",
				"url": "https://content.sportslogos.net/logos/67/1828/full/dominican_republic__logo_primary_2013_sportslogosnet-3062.png"
			},
		"Chinese Taipei":
			{
				"path": "Chinese Taipei.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1816/full/7444.gif")
			},
		"China":
			{
				"path": "China.png",
				"url": "https://content.sportslogos.net/logos/67/1815/full/china__logo_primary_2023_sportslogosnet-9201.png"
			},
		"Canada":
			{
				"path": "Canada.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1819/full/7425.gif")
			},
		"Australia":
			{
				"path": "Australia.png",
				"file": lambda: gif_url_to_png_base64("https://content.sportslogos.net/logos/67/1827/full/7435.gif")
			},			
		"Czechia":
			{
				"path": "Czechia.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/b/b5/Badge_of_the_Czech_Republic_national_baseball_team.svg/1280px-Badge_of_the_Czech_Republic_national_baseball_team.svg.png")
			},
		"Cuba":
			{
				"path": "Cuba.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/2/2c/Cuba_baseball_logo.svg/500px-Cuba_baseball_logo.svg.png")
			},
		"Colombia":
			{
				"path": "Colombia.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/5/5b/Logo_Seleccion_Beisbol_Colombia.svg/1280px-Logo_Seleccion_Beisbol_Colombia.svg.png")
			},
		"Brazil":
			{
				"path": "Brazil.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/4/42/Seleção_Brasileira_de_Beisebol.png")
			},
		"Nicaragua":
			{
				"path": "Nicaragua.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/7/76/Nicaragua_national_baseball_team_logo.png")
			},
		"Great Britain":
			{
				"path": "Great Britain.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/f/f2/Great_Britain_baseball_logo.png")
			},
		"Israel":
			{
				"path": "Israel.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/f/f7/Israel_national_baseball_team.png")
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack
	
def winter_package(game): #Caribbean Series, ABL
	base_graphics=[
			{
				"path": "Liga Mexicana del Pacifico.png",
				"file": lambda: url_to_base64("https://www.lmp.mx/assets/img/home2/logo_arco.png")
			},
			{
				"path": "Liga Venezuela Beisbol Profesional.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/0/0c/Venezuelan_Professional_Baseball_League_logo.png")
			},
			{
				"path": "Liga de Beisbol Profesional Roberto Clemente.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/d/d4/Liga_de_Béisbol_Profesional_Roberto_Clemente_logo.png")
			},
			{
				"path": "Liga de Beisbol Dominicano.png",
				"file": lambda: jpg_url_to_png_base64("https://img.mlbstatic.com/mlb-images/image/upload/t_2x1/t_w1536/mlb/wpydg4dwj6dqfpxoy4bt.jpg")
			},
			{
				"path": "Australian Baseball League.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/a/a5/Abl_baseball_logo.png")
			},
			{
				"path": "Caribbean Series.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/0/04/CaribbeanSeries.PNG")
			}			
			]
			
	team_graphics = {
		"Yaquis de Obregon": 
			{
				"path": "Yaquis de Obregon.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/c/cb/Yaquis_de_Obregon_logo.svg/1280px-Yaquis_de_Obregon_logo.svg.png")
			},
		"Venados de Mazatlan":
			{
				"path": "Venados de Mazatlan.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/e/ed/Venados_de_Mazatlán_wordmark.png")
			},
		"Tucson Baseball Team":
			{
				"path": "Tucson Baseball Team.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/0/08/Tucson_Baseball_Team_cap_insignia.png")
			},
		"Tomateros de Culiacan":
			{
				"path": "Tomateros de Culiacan.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/5/5d/Tomateros_de_Culiacán_Logo.png")
			},
		"Naranjeros de Hermosillo":
			{
				"path": "Naranjeros de Hermosillo.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/a/ad/Naranjeros_de_Hermosillo_logo.svg/1280px-Naranjeros_de_Hermosillo_logo.svg.png")
			},
		"Jaguares de Nayarit":
			{
				"path": "Jaguares de Nayarit.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/9/99/Jaguares_de_Nayarit_logo.svg/1280px-Jaguares_de_Nayarit_logo.svg.png")
			},
		"Charros de Jalisco":
			{
				"path": "Charros de Jalisco.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/3/37/Charros_de_Jalisco_logo.svg/250px-Charros_de_Jalisco_logo.svg.png")
			},
		"Caneros de los Mochis":
			{
				"path": "Caneros de los Mochis.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/c/ce/Cañeros_de_Los_Mochis_logo.svg/1280px-Cañeros_de_Los_Mochis_logo.svg.png")
			},
		"Algodoneros de Guasave":
			{
				"path": "Algodoneros de Guasave.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/a/a4/Algodoneros_de_Guasave_logo.png")
			},
		"Aguilas de Mexicali":
			{
				"path": "Aguilas de Mexicali.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/thumb/d/d1/Aguilas_de_Mexicali_logo.svg/1280px-Aguilas_de_Mexicali_logo.svg.png")
			},
		"Tiburones de la Guaira":
			{
				"path": "Tiburones de la Guaira.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/7/77/Tiburones_La_Guaira_logo.png")
			},
		"Navegantes del Magallanes":
			{
				"path": "Navegantes del Magallanes.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/e/e5/Insignia_de_los_Navegantes_del_Magallanes.svg/500px-Insignia_de_los_Navegantes_del_Magallanes.svg.png")
			},
		"Leones del Caracas":
			{
				"path": "Leones del Caracas.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/e/e2/Caracas_BBC_1942.png")
			},
		"Caribes de Anzoategui":
			{
				"path": "Caribes de Anzoategui.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/6/69/CaribesBBC_Anzoátegui_Logo.png")
			},
		"Cardenales de Lara":
			{
				"path": "Cardenales de Lara.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/c/ca/Cardenaleslogo.png")
			},			
		"Aguilas del Zulia":
			{
				"path": "Aguilas del Zulia.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/6/6d/Águilas_del_Zulia_team_logo.png")
			},
		"Tigres de Aragua":
			{
				"path": "Tigres de Aragua.png",
				"file": lambda: url_to_base64("https://www.tigresdearaguabbc.com/assets/img/site/logo-footer-v3.png")
			},
		"Bravos de Margarita":
			{
				"path": "Bravos de Margarita.png",
				"file": lambda: url_to_base64("https://www.bravosdemargarita.com/assets/img/site/logo.png")
			},
		"Senadores de San Juan":
			{
				"path": "Senadores de San Juan.png",
				"file": lambda: jpg_url_to_png_base64("https://upload.wikimedia.org/wikipedia/commons/c/c0/Senadores_de_San_Juan_logo.jpg")
			},
		"Cangrejeros de Santurce":
			{
				"path": "Cangrejeros de Santurce.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/4/48/Santurce_logo.png")
			},
		"Indios de Mayaguez":
			{
				"path": "Indios de Mayaguez.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/c/c8/Indios_de_Mayagüez_Logo.png")
			},
		"Criollos de Caguas":
			{
				"path": "Criollos de Caguas.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/0/04/Criollos_de_Caguas_logo.png")
			},
		"Leones de Ponce":
			{
				"path": "Leones de Ponce.png",
				"file": lambda: url_to_base64("https://ligapr.com/storage/images/teams/sFoSvuPUA8zfkZFIZrYTFpzRGorDlmlYKekHzGYn.png")
			},
		"Gigantes de Carolina":
			{
				"path": "Gigantes de Carolina.png",
				"file": lambda: url_to_base64("https://ligapr.com/storage/images/teams/RfjNRehSgODnb50HGeEVjmSmGk76mAdHz01CJ1qm.png")
			},
		"Toros del Este":
			{
				"path": "Toros del Este.png",
				"file": lambda: jpg_url_to_png_base64("https://upload.wikimedia.org/wikipedia/commons/5/51/Toros-del-Este-Logo.jpg")
			},
		"Tigres de Licey":
			{
				"path": "Tigres de Licey.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/0/07/Tigres_Del_Licey_Logo.png")
			},
		"Leones del Escogido":
			{
				"path": "Leones del Escogido.png",
				"file": lambda: jpg_url_to_png_base64("https://upload.wikimedia.org/wikipedia/commons/f/f2/Escogido_logo.jpg")
			},
		"Gigantes del Cibao":
			{
				"path": "Gigantes del Cibao.png",
				"file": lambda: jpg_url_to_png_base64("https://upload.wikimedia.org/wikipedia/commons/6/68/IsotipoRojoVino.jpg")
			},
		"Estrellas Orientales":
			{
				"path": "Estrellas Orientales.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/en/b/ba/Estrellas_Orientales_insignia.png")
			},
		"Aguilas Cibaenas":
			{
				"path": "Aguilas Cibaenas.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/1/19/Aguilas_Cibaenas_Logo.svg/1280px-Aguilas_Cibaenas_Logo.svg.png")
			},
		"Mexico":
			{
				"path": "Mexico.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/f/fc/Flag_of_Mexico.svg/1280px-Flag_of_Mexico.svg.png")
			},
		"Dominican Republic":
			{
				"path": "Dominican Republic.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/9/9f/Flag_of_the_Dominican_Republic.svg/1280px-Flag_of_the_Dominican_Republic.svg.png")
			},
		"Puerto Rico":
			{
				"path": "Puerto Rico.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/2/28/Flag_of_Puerto_Rico.svg/1280px-Flag_of_Puerto_Rico.svg.png")
			},
		"Venezuela":
			{
				"path": "Venezuela.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/0/06/Flag_of_Venezuela.svg/1280px-Flag_of_Venezuela.svg.png")
			},
		"Panama":
			{
				"path": "Panama.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/a/ab/Flag_of_Panama.svg/1280px-Flag_of_Panama.svg.png")
			},
		"Nicaragua":
			{
				"path": "Nicaragua.png",
				"file": lambda: url_to_base64("https://upload.wikimedia.org/wikipedia/commons/thumb/1/19/Flag_of_Nicaragua.svg/1280px-Flag_of_Nicaragua.svg.png")
			}
	}
		
	use_graphics=[
		resolve_entry(
			team_graphics[
				game['Away Team']
			]
		), 
		resolve_entry(
			team_graphics[
				game['Home Team']
			]
		)
	]
	base_graphics=[resolve_entry(x) for x in base_graphics]
	graphics_pack=base_graphics + use_graphics
	return graphics_pack